'''
kav module - Kaspersky support, version 0.3

(c) 2003-2018 Laszlo BALINT (blaci) <balint_laszlo(at)tiszanet.hu>
              Jan ONDREJ (SAL) <ondrejj(at)salstar.sk>
                                                                                
 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

'''

from avlib import *
import os,re

__all__ = ['kav', 'kavclient']

class kav(ascanner):
  '''
  Kaspersky Antivirus realscanner.

  This scanners is a realscanner, which can be used to scan
  for viruses. It parses Kaspersky antivirus output.
  
  Usage: kav(command=['/opt/kav/5.5/kav4unix/bin/kavscanner','-j3'])

  Where: command is a array which defines command with parameters.
  '''
  name='kav()'
  def __init__(self, command=['/opt/kav/5.5/kav4unix/bin/kavscanner','-j3']):
      self.command = command
  def create_args(self, files, dir=''):
      self.args = []
      for fn in files:
        if dir:
          self.args.append(safe.fn(os.path.join(dir, fn)))
        else:
          self.args.append(safe.fn(fn))
  def scanfile(self, files, dir='', args={}):
      level = 0.0
      detected = b''
      ret = []
      self.create_args(files, dir)
      pf = popen(self.command+self.args)
      pf.tocmd.close()
      rc = pf.wait()
      if rc==1:
        raise ScannerError('Unable to connect to aveserver.')
      elif rc==9:
        raise ScannerError('Some of the required parameters are missing from the command line.')
      # read lines from command
      while 1:
        line = pf.readline()
        if line=='': break
        re1 = re.search("(LINF|INF|SUSP)(ECTED|ICION) (.*)", line)
        if re1:
          debug.echo(3,"INFECTED: kav(): ", line)
          if re1.group(1) in ["INF","LINF"]: # if it is an indentified virus
            detected = re1.group(3)
            level += 1.0
          else:
            level += 0.75
            if detected==b'': # if it is suspected, but no ident. virus
              detected = re1.group(3)
        ret.append(line)
      pf.fromcmd.close()
      debug.echo(4, "KAV: RET: ", [[rc]], ret)
      return level, detected, ret

class kavclient(kav):
  '''
  Kaspersky antivirus client realscanner.

  This scanners is a realscanner, which can be used to scan
  for viruses. It parses Kaspersky antivirus client output.
  
  Usage: kavclient(socket_path='/var/run/aveserver',
                   command='/opt/kav/5.5/kav4mailservers/bin/aveclient',
                   chroot=True)

  Where: socket_path is a string, which defines path co aveserver socket
           inside chroot. (default: /var/run/aveserver)
         command is a string, which defines command to run.
           (default: /opt/kav/bin/aveclient)
         chroot is an string, which defines a prefix added to each filename.
           If you are not running aveserver in chroot path, set it
           to your CHROOT.
  '''
  name='kavclient()'
  def __init__(self, socket_path='/var/run/aveserver',
               command='/opt/kav/5.5/kav4mailservers/bin/aveclient',
               chroot=''):
      self.command = [command,'-s','-p']
      self.command_lastarg = socket_path
      self.chroot = chroot
  def create_args(self, files, dir=''):
      self.args = [safe.fn(self.command_lastarg)]
      for fn in files:
        if dir:
          self.args.append(self.chroot+os.path.join(dir, fn))
        else:
          self.args.append(self.chroot+fn)
