'''
ryporter.py - An virtual service for sagator's reporter.

(c) 2004-2016 Jan ONDREJ (SAL) <ondrejj(at)salstar.sk>

 This program is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 2 of the License, or
 (at your option) any later version.

'''

import re, socket
from aglib import *

__all__=['reporter']

## Report message contains from three variables. First part is message header.
## This part must contain email header described in RFC 2822.
## After this header you can define emails text.

_BEGIN='''\
From: SAGATOR mailer <%(SGUSER)s@%(MYHOSTNAME)s>
To: %(REPORT_RECIPIENT)s
Subject: Summary of undelivered emails: %(BEGIN)s - %(END)s
Precedence: bulk

This email is automaticcaly generated by SAGATOR.
It contains summary of emails, which has been rejected or dropped.

---------------------[ beginning of report ]--------------------------------
'''

## Each message contains more message bodys.
_BODY='''
%(COUNTER)s. Sender:     %(SENDER)s [%(SENTBY_IP)s]
     Subject:    %(SUBJECT)s
     Recipients: %(RECIPIENTS)s
     Date:       %(DATETIME)s
     Status:     %(VIRNAME)s %(STATUS)s [%(SCANNER_NAME)s=%(LEVEL)s]
     Qname:      %(QNAME)s
'''

## At end of message there is this text.
_END='''
--------------------------------------[ end of report ]---------------------
'''

class reporter(service):
  '''
  Reporter virtual service.
  
  This service is only a virtual service to configure parameters for
  reporter script.
  
  Usage: reporter(...parameters...)

  There you can define some parameters:
  "begin", "body", "end", "include", "exclude", "include_fx", "exclude_fx"
  and "groups".

  First three parameters are email templates. "begin" is report header.
  It must contain RFC2822 headers. "body" is message part displayed for each
  rejected/dropped message. "end" is report's tailer, added to message
  after all "body"s. See srv/reporter.py file for example.
  
  "webq" parameter must define base URL to webq() service. This parameter
  is autodetected as your server hostname + standard webq() root directory.
  A row with this URL is added to your standard body, if no user body
  defined.

  "groups" parameter can be used to define email groups, for example you
  can define to send reports only to admin for each domain on you server:
    reporter(groups=[
                     ['@mydomain1.com$', 'admin@mydomain1.com'],
                     ['@domain2.sk$', 'admin@domain2.sk'],
                     ['.', 'root@localhost'] # send other to root
                    ])
  It is possible to define empty string as target to ignore some records.
  
  Example: reporter(include='@mydomain.sk$')
  
  Groups are new in version 0.9.0.
  '''
  name='reporter()'
  INCLUDE='.'
  EXCLUDE='^$'
  INCLUDE_FX=re.compile(r'.').search
  EXCLUDE_FX=re.compile(r'^$').search
  BEGIN=_BEGIN
  BODY=_BODY
  GROUPS=[]
  END=_END
  def __init__(self,**args):
      self.init_args=args
      self.WEBQ_URL='https://%s/webq' % socket.gethostname()
      for key,value in list(args.items()):
        if key=='begin':
          self.BEGIN=value
        elif key=='body':
          self.BODY=value
        elif key=='end':
          self.END=value
        elif key=='include':
          self.INCLUDE_FX=re.compile(value, re.I).search
        elif key=='exclude':
          self.EXCLUDE_FX=re.compile(value, re.I).search
        elif key=='include_fx':
          self.EXCLUDE_FX=value
        elif key=='exclude_fx':
          self.INCLUDE_FX=value
        elif key=='groups':
          self.GROUPS=value
        elif key=='webq_url':
          self.WEBQ_URL=value.rstrip('/')
        else:
          raise TypeError('%s got an unexpected argument %s' % (self.name,key))
  def start(self):
      return []
